from svggen.api.component import Component
from svggen.api.composables.graph.Face import Face, Rectangle
from svggen.api.composables.GraphComposable import Graph
from svggen.api.ports.FacePort import FacePort
from svggen.api.ports.EdgePort import EdgePort
from svggen.api.ports.ModularPort import ModularPort
import svggen.utils.mymath as math

class RectBeam(Component):

  _test_params = {
    'length': 100,
    'width': 20,
    'depth': 50,
    'rangle': 90,
    'langle': 90,
    'phase': 2,
    'faces': None,
  }


  def define(self):
    self.addParameter("length")
    self.addParameter("width")
    self.addParameter("depth")
    self.addParameter("minlength", 0)
    self.addParameter("minwidth", 0)
    self.addParameter("mindepth", 0)
    self.addParameter("phase", 0)

    self.addParameter("angle")
    self.addParameter("rangle", 90)
    self.addParameter("langle", 90)

    self.addParameter("noflap", False)
    self.addParameter("faces")

    for i in range(4):
      self.addInterface("topedge%d" % i, EdgePort(self, None))
      self.addInterface("botedge%d" % i, EdgePort(self, None))
      self.addInterface("face%d" % i, FacePort(self, None, None))

    self.addInterface("tabedge", EdgePort(self, None))
    self.addInterface("slotedge", EdgePort(self, None))

    self.addInterface("modular", ModularPort(self, None))

    modular = {}
    modular["edges"] = {"FL": "topedge2",
                        "FR": "topedge0",
                        "BR": "botedge0",
                        "BL": "botedge2",}
    modular["faces"] = {"R": ["face0"],
                        "T": ["face1"],
                        "L": ["face2"],
                        "B": ["face3"],}
    modular["parameters"] = { "length": {"parameter": "length"},
                              "depth" : {"parameter": "width"},
                              "width" : {"parameter": "depth"},
                              "minlength": {"parameter": "minlength"},
                              "mindepth" : {"parameter": "minwidth"},
                              "minwidth" : {"parameter": "mindepth"} }

    self.setInterface("modular", ModularPort(self, modular))

  def modifyParameters(self):
    try:
      self.setParameter("width", max(self.getParameter("width"), self.getParameter("minwidth")))
    except KeyError:
      self.setParameter("width", self.getParameter("minwidth"))

    try:
      self.setParameter("depth", max(self.getParameter("depth"), self.getParameter("mindepth")))
    except KeyError:
      self.setParameter("depth", self.getParameter("mindepth"))

    try:
      self.setParameter("length", max(self.getParameter("length"), self.getParameter("minlength")))
    except KeyError:
      self.setParameter("length", self.getParameter("minlength"))

  def assemble(self):
    graph = Graph()

    try:
      rangle = 90 - self.getParameter("angle")
      langle = 90 - self.getParameter("angle")
    except KeyError:
      rangle = 90 - self.getParameter("rangle")
      langle = 90 - self.getParameter("langle")

    try:
      faces = self.getParameter("faces")
    except KeyError:
      faces = None

    length = self.getParameter("length")
    width = self.getParameter("width")
    depth = self.getParameter("depth")
    phase = self.getParameter("phase")

    rs = []
    rs.append(Rectangle("", width, length))
    rs.append(Face("", (
      (depth, math.tan(math.deg2rad(langle)) * depth),
      (depth, length - math.tan(math.deg2rad(rangle)) * depth),
      (0, length), (0,0)
    )))
    rs.append(Rectangle("", width, length - (math.tan(math.deg2rad(langle)) + math.tan(math.deg2rad(rangle))) * depth))
    rs.append(Face("", (
      (0, length), (0,0),
      (depth, math.tan(math.deg2rad(rangle)) * depth),
      (depth, length - math.tan(math.deg2rad(langle)) * depth),
    )))

    for i in range(phase):
      rs.append(rs.pop(0))

    fromEdge = None
    for i in faces or range(4):
      graph.attachFace(fromEdge, rs[i], "e3", prefix="r%d"%i, angle=90)
      fromEdge = 'r%d.e1' % i

    if faces is None:
      graph.addTab("r0.e3", "r3.e1", angle= 90, width=min(10, [depth, width][phase % 2]))

    self.composables["graph"] = graph

    #Define interfaces
    for i, n in enumerate(faces or range(4)):
      self.setInterface("topedge%d" % i, EdgePort(self, "r%d.e0" % n))
      self.setInterface("botedge%d" % i, EdgePort(self, "r%d.e2" % n))
      self.setInterface("face%d" % i, FacePort(self, graph, "r%d" % ((n-phase)%4)))

    self.setInterface("tabedge", EdgePort(self, fromEdge))
    self.setInterface("slotedge", EdgePort(self, "r%d.e3" % (faces or range(4))[0]))

if __name__ == "__main__":
  b = RectBeam()
  b._make_test()

