from svggen.api.component import Component
from svggen.utils.mymath import sin, pi, deg2rad, tan, cos, N
from svggen.api.composables.graph.Face import Face
from svggen.api.composables.GraphComposable import Graph
from svggen.api.composables.FunctionComposable import FunctionComposable
from svggen.api.ports.EdgePort import EdgePort
from svggen.api.ports.FacePort import FacePort

class Beam(Component):

  _test_params = {
    'length': 100,
    'beamwidth': 10,
    'shape': 3,
    'tangle': 45,
    'bangle': 90,
    'phase': 2,
  }

  def define(self):
    self.addParameter("length")
    #ANDY TODO: figure out what these keys were for and implement them more elegantly
    #self.addParameter("diameter")
    self.addParameter("beamwidth")

    self.addParameter("shape", 3)
    self.addParameter("phase", 0)

    self.addParameter("angle")
    self.addParameter("tangle", 90)
    self.addParameter("bangle", 90)

    self.addInterface("topedge", EdgePort(self, None))
    self.addInterface("botedge", EdgePort(self, None))
    for i in range(3):
      self.addInterface("face%d" % i, FacePort(self, None, None))

  def assemble(self):
    ### Assemble the object
    try:
      tangle = self.getParameter("angle")
      bangle = self.getParameter("angle")
    except KeyError:
      tangle = self.getParameter("tangle")
      bangle = self.getParameter("bangle")

    # XXX Hack: using numeric pi because symbolic pi hangs.  Should be using symbolic pi though
    try:
      d = self.getParameter("diameter")
      t = self.getParameter("diameter") * sin(N(pi) / self.getParameter("shape"))
    except KeyError:
      d = self.getParameter("beamwidth") / sin(N(pi) / self.getParameter("shape"))
      t = self.getParameter("beamwidth")

    length = self.getParameter("length")
    shape = self.getParameter("shape")
    phase = self.getParameter("phase")

    radius = d/2.
    dtheta = deg2rad(360. / shape)
    thetas = [ dtheta / 2. + dtheta * i for i in range(shape) ]

    thickness = 2 * radius * sin(dtheta / 2.)

    dl = [ radius * (1 - cos(t)) / tan(deg2rad(bangle)) for t in thetas ]
    dl = [ l - dl[-phase % shape] for l in dl ]
    dr = [ radius * (1 - cos(t)) / tan(deg2rad(tangle)) for t in thetas ]
    dr = [ r - dr[-phase % shape] for r in dr ]

    graph = Graph()
    angle = 360.0 / shape
    faces = []
    for i in range(len(thetas)):
      faces.append(Face('', ((thickness, dr[i]), (thickness, length - dl[i]), (0, length - dl[i-1]), (0, dr[i-1]))))
    for i in range(phase):
      faces.append(faces.pop(0))

    fromEdge = None
    for i in range(len(faces)):
      graph.attachFace(fromEdge, faces[i], 'e3', prefix="r%d"%i, angle = angle)
      fromEdge = "r%d.e1" % i

    if phase < 0:
      graph.addTab(fromEdge, faces[0].name + ".e3", angle = angle, width=thickness)
    else:
      graph.addTab(faces[0].name + ".e3", fromEdge, angle = angle, width=thickness)

    self.composables["graph"] = graph
    
    #Semantic constraints:
    self.addSemanticConstraint("0.9*beamwidth", "<", "length")
    self.addSemanticConstraint("-beamwidth", "<", "0.0")
    self.addSemanticConstraint("-length", "<", "0.0")

    # Assign interfaces
    self.setInterface("topedge", EdgePort(self, "r%d.e0" % (-phase % shape)))
    self.setInterface("botedge", EdgePort(self, "r%d.e2" % (-phase % shape)))
    for i in range(3):
      self.setInterface("face%d" % i, FacePort(self, graph, "r%d" % i))


    cf = FunctionComposable()
    cf.addInput(self, "topedge", default=0)
    cf.setOutput(self, "botedge", lambda topedge: length + topedge)
    self.composables["function"] = cf

if __name__ == "__main__":
  b = Beam()
  b._make_test()

  print b.getInterface("botedge").getValue()
  b.getInterface("topedge").setInputValue(-10)
  print b.getInterface("botedge").getValue()
