from collections import OrderedDict
from os.path import join
import sys

import yaml
import svggen.utils.mymath as math

from svggen import SVGGEN_DIR
from svggen.api.Parameterized import Parameterized
from svggen.api.Function import YamlFunction
from svggen.utils.utils import prefix as prefixString
from svggen.utils.utils import tryImport
from svggen.utils.io import load_yaml

def getSubcomponentObject(component, name=None):
    try:
        obj = tryImport(component, component)
        # XXX hack to get around derived components not having name parameter in their __init__
        c = obj()
        c.setName(name)
        return c
    except ImportError:
        c = Component(component)
        c.setName(name)
        return c


class Component(Parameterized):
    def __init__(self, yamlFile=None):
        self._name = None
        Parameterized.__init__(self)
        self.rebuild()
        if yamlFile:
            try:
                self.fromYaml(yamlFile)
            except IOError:
                self.fromYaml(join(SVGGEN_DIR, yamlFile))
        self.resolveSubcomponents()

    def getName(self):
      return self._name if self._name is not None else str(self.__class__)

    def setName(self, name):
      self._name = name

    def _make_test(self, **kwargs):
        if not hasattr(self, '_test_params'):
            raise NotImplementedError

        for key, val in self._test_params.iteritems():
            self.setParameter(key, val)

        name = self.__class__.__name__
        self.makeOutput('output/%s' % name, **kwargs)

    def rebuild(self):
        self.reset()
        self.define()

    def fromYaml(self, filename):
        definition = load_yaml(filename)

        # keys are (parameters, metadata, subcomponents, connections, interfaces)
        if "parameters" in definition:
          self.parameters = definition["parameters"]

        if "metadata" in definition:
          self.metadata = definition["metadata"]

        if "subcomponents" in definition:
          val = definition["subcomponents"]
          for k, v in val.iteritems():
            self.addSubcomponent(k, v["classname"], **v["kwargs"])
            self.subcomponents[k]["parameters"] = v["parameters"]

        if "connections" in definition:
          self.connections = definition["connections"]

        if "interfaces" in definition:
          val = definition["interfaces"]
          for k, v in val.iteritems():
            if isinstance(v, dict):
              self.inheritInterface(k, (v["subcomponent"], v["interface"]))
            else:
              self.interfaces[k] = v
          self.reinheritAllInterfaces()

    def reset(self):
        self.parameters = {}
        self.subcomponents = {}
        self.connections = {}
        self.interfaces = {}
        self.defaults = {}
        self.semanticConstraints = []

        self.components = {}
        self.composables = OrderedDict()

    def define(self):
        ### Override to define interfaces
        pass

    ###
    # DESIGN PHASE
    ###

    def addSubcomponent(self, name, classname, inherit=False, prefix = "", **kwargs):
        '''

        :param name: unique identifier to refer to this component by
        :type  name: str or unicode
        :param classname: code name of the subcomponent
                    should be python file/class or yaml name
        :type  classname: str or unicode
        '''
        # XXX will silently fail if subcomponent name is already taken?
        obj = getSubcomponentObject(classname, self.getName() + '.' + name)
        sc = {"classname": classname, "parameters": {}, "kwargs": kwargs, "object": obj}
        self.subcomponents.setdefault(name, sc)

        if inherit:
            if prefix == "":
                prefix = name

            for key, value in obj.parameters.iteritems():
                # inherit = True : inherit all parameters
                if inherit is True or key in inherit:
                    try:
                        self.addParameter(prefixString(prefix, key), value, **obj.metadata[key])
                    except KeyError:
                        # It's ok if we try to add a parameter that already exists
                        pass
                    self.addConstraint((name, key), prefixString(prefix, key))
            # XXX also inherit interfaces?

        return self

    def delSubcomponent(self, name):
        toDelete = []

        # delete edges connecting components
        for connName, ((fromComp, _), (toComp, _), _) in self.connections.iteritems():
            if name in (fromComp, toComp):
                toDelete.append(connName)
        for connName in toDelete:
            self.connections.pop(connName)

        self.subcomponents.pop(name)

    def addConstraint(self, (subComponent, parameterName), inputs, function=None):
        # XXX silently overwrites existing constraints, is that ok?
        if function:
          self.subcomponents[subComponent]["parameters"][parameterName] = {"function": function, "parameter": inputs}
        else:
          self.subcomponents[subComponent]["parameters"][parameterName] = {"parameter": inputs}

    def addConstConstraint(self, (subComponent, parameterName), value):
        self.subcomponents[subComponent]["parameters"][parameterName] = value

    def delConstraint(self, subComponent, parameterName):
        self.subcomponents[subComponent]["parameters"].pop(parameterName)

    def addInterface(self, name, val):
        if name in self.interfaces:
            raise ValueError("Interface %s already exists" % name)
        self.interfaces.setdefault(name, val)
        return self

    def inheritAllInterfaces(self, subcomponent, prefix=""):
        obj = self.subcomponents[subcomponent]["object"]
        if prefix == "":
          prefix = name
        for name, value in obj.interfaces.iteritems():
          self.inheritInterface(prefixString(prefix, name), (subcomponent, name))
        self.reinheritAllInterfaces()
        return self

    def inheritInterface(self, name, (subcomponent, subname)):
        if name in self.interfaces:
            raise ValueError("Interface %s already exists" % name)
        value = {"subcomponent": subcomponent, "interface": subname}
        obj = self.subcomponents[subcomponent]["object"]
        iobj = obj.getInterface(subname, transient=True).inherit(self, subcomponent)
        if iobj:
          value["object"] = iobj
        self.interfaces.setdefault(name, value)

        return self

    # XXX kinda hackish?
    def reinheritAllInterfaces(self):
      for name, value in self.interfaces.iteritems():
        if isinstance(value, dict):
          subc = value["subcomponent"]
          subi = value["interface"]
          obj = self.subcomponents[subc]["object"]
          iobj = obj.getInterface(subi, transient=True).inherit(self, subc)
          if iobj:
            value["object"] = iobj
          self.interfaces.setdefault(name, value)

    def addConnection(self, fromInterface, toInterface, name=None, **kwargs):
        if name is None:
          for i in range(len(self.connections)+1):
            name = "connection%d" % i
            if name not in self.connections:
              break
        for k, v in kwargs.iteritems():
          try:
            kwargs[k] = v.toYamlObject()
          except AttributeError:
            pass
        self.connections.setdefault(name, [fromInterface, toInterface, kwargs])

    def delConnection(self, name):
      self.connections.pop(name)

    def getConnections(self, component1, component2=None):
      keys = []
      for connName, (fromInterface, toInterface, kwargs) in self.connections.iteritems():
        if component1 in [fromInterface[0], toInterface[0]]:
          if component2 is None or component2 in [fromInterface[0], toInterface[0]]:
            keys.append(connName)
      return keys
          

    '''
    # TODO : delete Interface
    # XXX : remove constraints that involve this parameter?
    def delParameter(self, name):
    '''

    def toYaml(self, filename):

        subcomponents = {}
        for k, v in self.subcomponents.iteritems():
          subcomponents[k] = {"classname": v["classname"], "parameters": v["parameters"], "kwargs": v["kwargs"]}

        interfaces = {}
        for k, v in self.interfaces.iteritems():
          if isinstance(v, dict):
            interfaces[k] = {"subcomponent": v["subcomponent"], "interface": v["interface"]}
          else:
            interfaces[k] = v

        definition = {
            "parameters" : self.parameters,
            "metadata" : self.metadata,
            "subcomponents" : subcomponents,
            "connections" : self.connections,
            "interfaces" : interfaces,
        }

        with open(join(SVGGEN_DIR, filename), "w") as fd:
            yaml.safe_dump(definition, fd)

    ###
    # GETTERS AND SETTERS
    ###

    def addComponent(self, name, obj, classname, **kwargs):
        # XXX will silently fail to set if name is already taken?
        # what about
        # if name in self.components:
        #     raise something
        # else:
        #     self.cponents[name] = (obj, classname)
        self.components.setdefault(name, (obj, classname, kwargs))
        return self

    def getComponent(self, name):
        return self.components[name][0]

    def setSubParameter(self, c, n, v):
        self.getComponent(c).setParameter(n, v)

    def getInterfaces(self, component, name, transient=False):
        return self.getComponent(component).getInterface(name, transient)

    def getInterface(self, name, transient=False):
        c = self.interfaces[name]

        if isinstance(c, dict):
            if "object" in c and c["object"]: 
              return c["object"]
            
            subc = c["subcomponent"]
            subi = c["interface"]
            i = self.getInterfaces(subc, subi, transient)
            if not transient:
              c["object"] = i
            return i
        else:
          return c

    def setInterface(self, n, v):
        if n in self.interfaces:
            self.interfaces[n] = v
        else:
            raise KeyError("Interface %s not initialized" % n)
        return self

    ###
    # ASSEMBLY PHASE
    ###

    def assemble(self):
        ### Override to combine components' drawings to final drawing
        pass

    def append(self, name, prefix, **kwargs):
        component = self.getComponent(name)

        allPorts = set()
        for key in component.interfaces:
          try:
            allPorts.update(component.getInterface(key))
          except TypeError:
            # interface is not iterable, i.e. a single port
            allPorts.add(component.getInterface(key))
        for port in allPorts:
          port.prefix(prefix)

        for (key, composable) in component.composables.iteritems():
            self.composables[key].append(composable, prefix, **kwargs)

    def attach(self, (fromName, fromPort), (toName, toPort), kwargs):
        for (key, composable) in self.composables.iteritems():
            try:
                composable.attachInterfaces(self.getInterfaces(fromName, fromPort),
                                            self.getInterfaces(toName, toPort),
                                            kwargs)
            except:
                print "Error in attach:"
                print (fromName, fromPort),
                print self.getInterfaces(fromName, fromPort).toString()
                print (toName, toPort),
                print self.getInterfaces(toName, toPort).toString()
                raise

    ###
    # BUILD PHASE
    ###

    def modifyParameters(self):
        # Override to manually specify how parameters get set during build
        pass

    def resolveSubcomponents(self):
        for (name, sc) in self.subcomponents.iteritems():
            c = sc["classname"]
            obj = sc["object"]
            try:
              kwargs = sc["kwargs"]
            except KeyError:
              kwargs = {}
            self.addComponent(name, obj, c, **kwargs)

    def evalConstraints(self):
        for subComponent in self.subcomponents.iterkeys():
            for (parameterName, obj) in self.subcomponents[subComponent]["parameters"].iteritems():
                try:
                  x = YamlFunction(obj).eval(self)
                except Exception as e:
                  print "Error trying to evaluate", obj
                  print e
                  raise
                if x is not None:
                  self.setSubParameter(subComponent, parameterName, x)

    # Append composables from all known subcomponents
    # (including ones without explicitly defined connections)
    def evalComponents(self):
        for (name, sc) in self.components.iteritems():
            obj = sc[0]
            classname = sc[1]
            try: 
              kwargs = sc[2]
            except IndexError:
              kwargs = {}

            try:
                obj.make()
                for (key, composable) in obj.composables.iteritems():
                    if key not in self.composables:
                        self.composables[key] = composable.new()
                self.append(name, name, **kwargs)
            except:
                print "Error in subclass %s, instance %s" % (classname, name)
                raise
        # Let composables know what components exist
        # TODO remove this when we have a better way of letting composables
        # know about components that have no ports (ex Bluetooth module driver)
        for (key, composable) in self.composables.iteritems():
          for (name, sc) in self.components.iteritems():
            composable.addComponent(sc[0])

    def evalInterfaces(self):
      for (name, value) in self.interfaces.iteritems():
        for (key, composble) in self.composables.iteritems():
          if value is not None:
            composble.addInterface(self.getInterface(name))

    def evalConnections(self):
        for ((fromComponent, fromPort), (toComponent, toPort), kwargs) in self.connections.values():
            for k, v in kwargs.iteritems():
              kwargs[k] = YamlFunction(v).eval(self)

            self.attach((fromComponent, fromPort),
                        (toComponent, toPort),
                        kwargs)

    def organizeConnections(self,**kwargs):
        '''
        This function would reorganize the order of connection. It should works with edge, but it does not work with angles yet.

        This code is yet to be debugged and optimized.

        Here are the four cases it aims to deal with:
        case1: addConnection((non-existingFaceA,non-existingEdgeA),(non-existingFaceB,non-existingEdgeB),angle=X)
        The user is creating a new connection with two faces.
        organizeConnection would insert a k,v pair in the dictionary, with k=non-existingEdgeB, v=non-existingEdgeA

        case2: addConnection((non-existingFaceA,non-existingEdgeA),(existingFaceB,existingEdgeB),angle=X)
        The user is adding a new face to an existing connection.

            subcase1:  existingEdgeB is a key in the dictionary
            subcase2: existingEdgeB is a value in the dictionary

        case3: addConnection((existingFaceA,existingEdgeA),(non-existingFaceB,non-existingEdgeB),angle=X)
        The user is adding a new face to an existing connection, but in the reverse order compared with case2.

            subcase1:  existingEdgeA is a key in the dictionary
            subcase2: existingEdgeA is a value in the dictionary

        case4: addConnection((existingFaceA,existingEdgeA),(existingFaceB,existingEdgeB),angle=X)
        The user seems to be updating an existing pair. I would not deal with it yet.

        No, at second thought, it is a bug!!!!!How to fix it? Hmm...


        :return:
        '''
        if kwargs['thickness']==0:
            return

        def organizeHelper(toSurfaces, edge):
            for k, v in toSurfaces.iteritems():
                if edge == k:
                    return ["indict","iskey", k]
                for each_edge in v:
                    if edge in each_edge:
                        return ["indict","isvalue", k, each_edge[1]]
            return ["notindict", None]

        toSurfaces = {}
        for k, v in self.connections.iteritems():
            list0 = organizeHelper(toSurfaces, v[0])
            list1 = organizeHelper(toSurfaces, v[1])
            if list0[0]=="notindict" and list1[0]=="notindict": #case1
                toSurfaces.setdefault(v[1], [[v[0],v[2]['angle']]])

            elif list0[0] == "notindict" and list1[0] == "indict": #case2
                if list1[1]=="iskey": #subcase1
                    toSurfaces[v[1]].append([v[0], v[2]['angle']])
                elif list1[1]=="isvalue": #subcase2
                    v[1]=list1[2]
                    v[2]['angle']=list1[3]+v[2]['angle']
                    toSurfaces[v[1]].append([v[0],v[2]['angle']])

            elif list0[0] == "indict" and list1[0] == "notindict": #case3
                if list0[1]=="iskey": #subcase1
                    v[0],v[1]=v[1],v[0]
                    v[2]['angle'] = -v[2]['angle']
                    toSurfaces[v[1]].append([v[0], v[2]['angle']])
                elif list0[1]=="isvalue": #subcase2
                    v[0]=v[1]
                    v[1]=list0[2]
                    v[2]['angle']=list0[3]-v[2]['angle']
                    toSurfaces[v[1]].append([v[0],v[2]['angle']])

            elif list0[0] == "indict" and list1[0] == "indict":  # case 4
                if list0[1]=="iskey" and list1[1]=="iskey":
                    pairs=toSurfaces.get(list0[2])
                    for each_connected_face in pairs:
                        for k1, v1 in self.connections.iteritems():
                            if v1[0]==each_connected_face[0] and v1[1]==list0[2]:
                                v1[1]=list1[2]
                                v1[2]['angle']=v1[2]['angle']+v[2]['angle']
                                toSurfaces[list1[2]].append([v1[0], v1[2]['angle']])
                    toSurfaces[list1[2]].append([v[0], v[2]['angle']])
                    toSurfaces.pop(list0[2])
                elif list0[1]=="isvalue" and list1[1]=="iskey":
                    offset_angle=list0[3]
                    for each_connected_face in toSurfaces.get(list0[2]):
                        for k1, v1 in self.connections.iteritems():
                            if v1[0]==each_connected_face[0] and v1[1]==list0[2]:
                                v1[1]=list1[2]
                                v1[2]['angle']=v1[2]['angle']+v[2]['angle']-offset_angle
                                toSurfaces[list1[2]].append([v1[0], v1[2]['angle']])
                    v[0]=list0[2]
                    v[2]['angle']=v[2]['angle']-offset_angle
                    toSurfaces[v[1]].append([v[0], v[2]['angle']])
                    toSurfaces.pop(list0[2])
                elif list0[1]=="iskey" and list1[1]=="isvalue":
                    offset_angle = list1[3]
                    for each_connected_face in toSurfaces.get(list0[2]):
                        for k1, v1 in self.connections.iteritems():
                            if v1[0]==each_connected_face[0] and v1[1]==list0[2]:
                                v1[1]=list1[2]
                                v1[2]['angle']=v1[2]['angle']+v[2]['angle']+offset_angle
                                toSurfaces[list1[2]].append([v1[0], v1[2]['angle']])
                    v[1] = list1[2]
                    v[2]['angle'] = v[2]['angle'] + offset_angle
                    toSurfaces[v[1]].append([v[0], v[2]['angle']])
                    toSurfaces.pop(list0[2])
                elif list0[1]=="isvalue" and list1[1]=="isvalue":
                    offset_angle1 = list0[3]
                    offset_angle2 = list1[3]
                    for each_connected_face in toSurfaces.get(list0[2]):
                        for k1, v1 in self.connections.iteritems():
                            if v1[0]==each_connected_face[0] and v1[1]==list0[2]:
                                v1[1]=list1[2]
                                v1[2]['angle']=v1[2]['angle']+v[2]['angle']-offset_angle1+offset_angle2
                                toSurfaces[list1[2]].append([v1[0], v1[2]['angle']])
                    v[0]=list0[2]
                    v[1]=list1[2]
                    v[2]['angle']=v[2]['angle']-offset_angle1+offset_angle2
                    toSurfaces[v[1]].append([v[0], v[2]['angle']])
                    toSurfaces.pop(list0[2])

    def make(self):
        self.modifyParameters()
        self.resolveSubcomponents()
        self.evalConstraints()

        self.evalComponents()    # Merge composables from all subcomponents and tell them my components exist
        self.evalInterfaces()    # Tell composables that my interfaces exist
        self.evalConnections()   # Tell composables which interfaces are connected
        self.assemble()

    ###
    # OUTPUT PHASE
    ###

    def makeComponentHierarchy(self):
        self.resolveSubcomponents()
        hierarchy = {}
        for n, sc in self.components.iteritems():
            sub = sc[0]
            c = sc[1]
            hierarchy[n] = {"class":c, "subtree":sub.makeComponentHierarchy()}
        return hierarchy

    def makeComponentTree(self, fn, root="Root"):
        import pydot
        graph = pydot.Dot(graph_type='graph')
        mynode = pydot.Node(root, label = root)
        self.recurseComponentTree(graph, mynode, root)
        graph.write_png(fn)

    def recurseComponentTree(self, graph, mynode, myname):
        import pydot
        self.resolveSubcomponents()
        for n, sc in self.components.iteritems():
            sub = sc[0]
            c = sc[1]
            fullstr = myname + "/" + n
            subnode = pydot.Node(fullstr, label = c + r"\n" + n)
            graph.add_node(subnode)
            edge = pydot.Edge(mynode, subnode)
            graph.add_edge(edge)
            sub.recurseComponentTree(graph, subnode, fullstr)

    def makeOutput(self, filedir=".", **kwargs):
        def kw(arg, default=False):
            if arg in kwargs:
                return kwargs[arg]
            return default

        if kw("testparams"):
            if not hasattr(self, '_test_params'):
                raise NotImplementedError
            for key, val in self._test_params.iteritems():
                self.setParameter(key, val)

        print "Compiling robot designs to directory", filedir, '...'
        sys.stdout.flush()
        if kw("remake", True):
            self.organizeConnections(**kwargs)
            self.make()
        print "done."

        # XXX: Is this the right way to do it?
        import os
        try:
            os.makedirs(filedir)
        except:
            pass

        # Process composables in some ordering based on type
        orderedTypes = ['electrical', 'ui', 'code'] # 'code' needs to know about pins chosen by 'electrical', and 'code' needs to know about IDs assigned by 'ui'
        # First call makeOutput on the ones of a type whose order is specified
        for composableType in orderedTypes:
            if composableType in self.composables:
                kwargs["name"] = composableType
                self.composables[composableType].makeOutput(filedir, **kwargs)
        # Now call makeOutput on the ones whose type did not care about order
        for (composableType, composable) in self.composables.iteritems():
            if composableType not in orderedTypes:
                kwargs["name"] = composableType
                self.composables[composableType].makeOutput(filedir, **kwargs)

        if kw("tree"):
            print "Generating hierarchy tree... ",
            sys.stdout.flush()
            self.makeComponentTree(filedir + "/tree.png")
            print "done."
        print

        print "Happy roboting!"

    ###
    # OTHER STUFF
    # (probably obsolete)
    ###

    def sympyicize(self, param):
        try:
            default = self.getParameter(param)
        except KeyError:
            default = 10.0 #ANDYTODO: LOL
        self.setParameter(param, math.Symbol(param, positive=True))
        self.defaults[param] = default #TODO: this is hacky
    
    def addSemanticConstraint(self, lhs, op, rhs):
        self.semanticConstraints.append([lhs, op, rhs])

