class Parameterized(object):
  """
  Like a dictionary k/v store, but we require special syntax constructs
  to set/update keys
  """
  def __init__(self):
    self.parameters = {}
    self.defaults = {}
    self.metadata = {}


  def addParameter(self, name, value=None, **kwargs):
    """
    Adds a k/v pair to the internal store if the key has not been added before
    Raises KeyError if the key has been added before
    """
    if name in self.parameters:
      raise KeyError("Parameter %s already exists on object %s" % (name, str(self)))
    self.parameters.setdefault(name, value)
    self.defaults[name] = value
    self.metadata[name] = kwargs
    self.metadata[name].update({"default": value})


  def setParameter(self, n, v):
    """
    Sets a k/v pair to the internal store if the key has been added previously
    Raises KeyError if the key has not been added before
    """
    if n in self.parameters:
      self.parameters[n] = v
      self.defaults[n] = v
    else:
      raise KeyError("Parameter %s not initialized on object %s" % (n, str(self)))


  def getParameter(self, name, strict=True):
    """
    Retrieves the parameter value with the given name
    Raises KeyError if the key is not been set
    """
    if strict and self.parameters[name] is None:
      raise KeyError("Parameter %s not yet set on object %s" % (name, str(self)))
    return self.parameters[name]


  def getParameterInfo(self):
    """
    Retrieves the parameter metadata info
    """
    return self.metadata


  def hasParameter(self, name):
    return name in self.parameters


  def delParameter(self, name):
    self.parameters.pop(name)
    self.defaults.pop(name)
    self.metadata.pop(name)

