#include "robotLibrary.h"

int servoPins[numServos];
int servoOffsets[numServos];
char bluetoothData[50];
bool bluetoothConnected;
bool validBluetoothData;
long PWMFrequency[3];
PROGMEM int const controllerPins[] = {-1, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1};
PROGMEM int const pinTypes[] = {-1, SI, -1, -1, -1, -1, -1, -1, -1, -1, -1, SI, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1};
PROGMEM int const protocol[] = {PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_BLUETOOTH, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_BLUETOOTH, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT, PROTOCOL_DIRECT};
int dataMapping[NUM_DATA_OUTPUTS][DATA_OUTDEGREE] = {{0}, {4}, {10}, {14}};
int dataOutputIDs[NUM_DATA_OUTPUTS] = {7, 9, 17, 19};
bool autoPoll[NUM_DATA_OUTPUTS][DATA_OUTDEGREE] = {{false}, {false}, {false}, {false}};
char outputData[50];
bool validGetData;


bool setSpeed(int servoNum, int speed)
{
  if(speed <= 5 && speed >= -5)
    speed = 0;
  if(speed == 0)
    setPWM(servoPins[servoNum], 0);
  else
    setPWM(servoPins[servoNum], speedToDuty(speed + servoOffsets[servoNum], servoPins[servoNum]));
  robotPrintDebug("Set servo ");
  robotPrintDebug(servoNum);
  robotPrintDebug(" to speed  "); robotPrintlnDebug(speed);
  return true;
}

void calibrateServo(int servoNum)
{
  getServoOffsets();
  int eepromAddress = 0;
  char input = 'x';
  Serial.begin(9600);
  robotPrintlnDebug("----------------------------");
  robotPrintlnDebug("Send \'+\' or \'.\' to increase the calibration value");
  robotPrintlnDebug("Send \'-\' or \',\' to decrease the calibration value");
  robotPrintlnDebug("Send \'s\' to save the calibration value");
  robotPrintlnDebug("----------------------------");
  robotPrintDebug("Calibrating servo "); robotPrintlnDebug(servoNum);
  while(input != 's')
  {
    robotPrintDebug("\tCalibration value: "); robotPrintlnDebug(servoOffsets[servoNum]);
    setPWM(servoPins[servoNum], speedToDuty(0 + servoOffsets[servoNum], servoPins[servoNum]));
    while(!Serial.available()) {}
    input = Serial.read();
    switch(input)
    {
      case '+':
      case '.': servoOffsets[servoNum]++; break;
      case '-':
      case ',': servoOffsets[servoNum]--; break;
      case 's': eepromWriteInt(2*servoNum, servoOffsets[servoNum]); break;
    }
  }
  robotPrintlnDebug("----------------------------");
  robotPrintlnDebug("Calibration complete!");
  delay(1500);
}

void getServoOffsets()
{
  for(int i = 0; i < numServos; i++)
  {
    servoOffsets[i] = eepromReadInt(2*i);
    if(servoOffsets[i] > 90 || servoOffsets[i] < -90)
      servoOffsets[i] = 0;
  }
}

void eepromWriteInt(int address, int value)
{
  union u_tag
  {
    byte b[2];        //assumes 2 bytes in an int
    int INTtime;
  }
  time;
  time.INTtime=value;

  EEPROM.write(address  , time.b[0]);
  EEPROM.write(address+1, time.b[1]);
}

int eepromReadInt(int address)
{
  union u_tag
  {
    byte b[2];
    int INTtime;
  }
  time;
  time.b[0] = EEPROM.read(address);
  time.b[1] = EEPROM.read(address+1);
  return time.INTtime;
}

int speedToDuty(double speed, int pin)
{
  double pwmPeriod = 1000.0/(double)getPWMFrequency(pin);
  double pulseWidth = (speed+100)/200.0 * (1.2 - 0.3) + 0.3;
  return pulseWidth / pwmPeriod * 255;
}

bool getBluetoothData()
{
  if(!bluetoothAvailable())
    return false;
  if(validBluetoothData)
  {
    for(int i = 0; i < 50; i++)
      bluetoothData[i] = '\0';
    validBluetoothData = false;
  }
  int index = length(bluetoothData);
  while(bluetoothAvailable() && !validBluetoothData)
  {
    bluetoothData[index++] = getBluetoothChar();
    validBluetoothData = (bluetoothData[index-1] == '\0');
  }
  if(validBluetoothData)
  {
    // If it is a heartbeat, respond to it now
    if(equals(bluetoothData, "?"))
    {
      sendBluetoothData("?");
      bluetoothData[0] = '\0';
      validBluetoothData = false;
      return false;
    }
    robotPrintDebug("Got BT data <"); robotPrintDebug(bluetoothData); robotPrintlnDebug(">");
  }
  return validBluetoothData;
}
//bool getBluetoothData()
//{
//  bluetoothData[0] = '\0';
//  if(!bluetoothAvailable())
//    return false;
//  int timeout = 50;
//  int index = 0;
//  bool terminated = false;
//  unsigned long start = millis();
//  while(!terminated && millis() - start < timeout)
//  {
//    while(!bluetoothAvailable() && millis() - start < timeout);
//    bluetoothData[index++] = getBluetoothChar();
//    start = millis();
//    terminated = (bluetoothData[index-1] == '\0');
//  }
//  if(index > 0 && bluetoothData[index-1] != '\0')
//    bluetoothData[index] = '\0';
//  // If it is a heartbeat, respond to it now
//  if(equals(bluetoothData, "?"))
//  {
//    sendBluetoothData("?");
//    bluetoothData[0] = '\0';
//    return false;
//  }
//  robotPrintDebug("Got BT data <"); robotPrintDebug(bluetoothData); robotPrintlnDebug(">");
//  return true;
//}

void sendBluetoothData(const char* data)
{
  int index = 0;
  for(; index < length(data); index++)
    sendBluetoothChar(data[index]);
  if(data[index-1] != '\0')
    sendBluetoothChar('\0');
  robotPrintDebug("Sent BT data <"); robotPrintDebug(data); robotPrintlnDebug(">");
}

void processBluetoothData()
{
  if(!getBluetoothData())
    return;
  if(indexOf(bluetoothData, uiRequest) >= 0)
  {
    sendUIDescription();
    return;
  }
  // Parse data
  bool isRequest = false;
  char data[10];  data[0] = '\0';
  if(indexOf(bluetoothData, dataCommand) >= 0)
    isRequest = false;
  else if(indexOf(bluetoothData, dataRequest) >= 0)
    isRequest = true;
  else
    return;
  bluetoothConnected = true;
  char outputIDChar[3]; outputIDChar[0] = '\0';
  char inputIDChar[3]; inputIDChar[0] = '\0';
  int index = 0;
  int btLength = length(bluetoothData);
  for(; index < btLength && bluetoothData[index] != '$'; index++);
  index++; // get passed first dollar sign
  if(!isRequest)
  {
    int dataIndex = 0;
    while(index < btLength && bluetoothData[index] != '$')
      data[dataIndex++] = bluetoothData[index++];
    data[dataIndex] = '\0';
    index++; // get passed second dollar sign
  }
  int idIndex = 0;
  while(index < btLength && bluetoothData[index] != '$')
    outputIDChar[idIndex++] = bluetoothData[index++];
  outputIDChar[idIndex] = '\0';
  index++;
  idIndex = 0;
  while(index < btLength && bluetoothData[index] != '$')
    inputIDChar[idIndex++] = bluetoothData[index++];
  inputIDChar[idIndex] = '\0';

  int outputID = length(outputIDChar) > 0 ? atoi(outputIDChar) : -1;
  int inputID = length(inputIDChar) > 0 ? atoi(inputIDChar) : -1;

  robotPrintDebug("\tgot data <"); robotPrintDebug(data); robotPrintlnDebug(">");
  robotPrintDebug("\tgot output <"); robotPrintDebug(outputIDChar); robotPrintDebug("> -> "); robotPrintlnDebug(outputID);
  robotPrintDebug("\tgot input <"); robotPrintDebug(inputIDChar); robotPrintDebug("> -> "); robotPrintlnDebug(inputID);

  if(isRequest)
  {
    getData(outputID);
    if(validGetData)
    {
      char toSend[50]; toSend[0] = '\0';
      strcpy(dataCommand, toSend, 50);
      concat(toSend, "$", toSend, 50);
      concat(toSend, outputData, toSend, 50);
      concat(toSend, "$", toSend, 50);
      concatInt(toSend, outputID, toSend, 50);
      concat(toSend, "$", toSend, 50);
      concat(toSend, inputIDChar, toSend, 50);
      sendBluetoothData(toSend);
    }
  }
  else
  {
    if(inputID >= 0 && outputID < 0)
      processData(data, inputID);
    else if(inputID >= 0 && outputID >= 0)
      processData(data, outputID, inputID);
    else if(outputID >= 0)
    {
      // Find index in array of this output ID
      for(int i = 0; i < NUM_DATA_OUTPUTS; i++)
      {
        if(dataOutputIDs[i] == outputID)
          processData(data, outputID, dataMapping[i], DATA_OUTDEGREE);
      }
    }
  }
}

bool isBluetoothConnected()
{
  return bluetoothConnected;
}

void setPinMode(int pin, int mode)
{
  switch(mode)
  {
    case DI: pinMode(pin, INPUT_PULLUP); break;
    case DO: pinMode(pin, OUTPUT); break;
    case AI: break;
    case AO:
    case PO: pinMode(pin, OUTPUT); break;
    case SI: break;
    case SO:
      pinMode(pin, OUTPUT);
      switch(pin)
      {
        case 5:
        case 6:
          PWMFrequency[0] = setPWMFrequency(pin, 980); break;
        case 9:
        case 10:
          PWMFrequency[1] = setPWMFrequency(pin, 480); break;
        case 3:
        case 11:
          PWMFrequency[2] = setPWMFrequency(pin, 480); break;
      }
      break;
  }
}

long getPWMFrequency(int pinIndex)
{
  switch(getControllerPin(pinIndex))
  {
    case 5:
    case 6:
      return PWMFrequency[0];
    case 9:
    case 10:
      return PWMFrequency[1];
    case 3:
    case 11:
      return PWMFrequency[2];
  }
}

int setPWMFrequency(int pin, long frequency)
{
  byte mode;
  long baseFrequency;
  if(pin == 3 || pin == 9 || pin == 10 || pin == 11)
      baseFrequency = 31250;
  if(pin == 5 || pin == 6)
      baseFrequency = 62500;
  long error = baseFrequency;
  int divisor = 1;
  if(pin == 5 || pin == 6 || pin == 9 || pin == 10)
  {
    int divisors[] = {1,8,64,256,1024};
    for(int i = 0; i < 5; i++)
    {
      long newError = frequency - baseFrequency / divisors[i];
      newError *= newError < 0 ? -1 : 1;
      if(newError < error)
      {
        error = newError;
        divisor = divisors[i];
      }
    }
    robotPrintDebug("divisor: "); robotPrintlnDebug(divisor);
    robotPrintDebug("frequency: "); robotPrintlnDebug(baseFrequency / divisor);
    robotPrintDebug("error: "); robotPrintlnDebug(error);
    switch(divisor)
    {
      case 1: mode = 0x01; break;
      case 8: mode = 0x02; break;
      case 64: mode = 0x03; break;
      case 256: mode = 0x04; break;
      case 1024: mode = 0x05; break;
      default: return -1;
    }
    if(pin == 5 || pin == 6)
      TCCR0B = TCCR0B & 0b11111000 | mode;
    else
      TCCR1B = TCCR1B & 0b11111000 | mode;
  }
  else if(pin == 3 || pin == 11)
  {
    int divisors[] = {1,8,32,64,128,256,1024};
    for(int i = 0; i < 7; i++)
    {
      long newError = frequency - baseFrequency / divisors[i];
      newError *= newError < 0 ? -1 : 1;
      if(newError < error)
      {
        error = newError;
        divisor = divisors[i];
      }
    }
    switch(divisor)
    {
      case 1: mode = 0x01; break;
      case 8: mode = 0x02; break;
      case 32: mode = 0x03; break;
      case 64: mode = 0x04; break;
      case 128: mode = 0x05; break;
      case 256: mode = 0x06; break;
      case 1024: mode = 0x7; break;
      default: return -1;
    }
    TCCR2B = TCCR2B & 0b11111000 | mode;
  }
  return baseFrequency / divisor;
}

void robotSetup()
{
  robotPrintSetupDebug(9600);
  // Set each pin to the correct mode
  for(int pinIndex = 0; pinIndex < numPins; pinIndex++)
  {
    int controllerPin = getControllerPin(pinIndex);
    if(controllerPin >= 0 && getPinMateType(pinIndex) >= 0)
      setPinMode(controllerPin, getPinMateType(pinIndex));
  }
  servoPins[0] = 1;
  getServoOffsets();
  //calibrateServo(0);
  setSpeed(0, 0);
  bluetoothConnected = false;
  validBluetoothData = false;
  bluetoothData[0] = '\0';
  servoPins[1] = 11;
  getServoOffsets();
  //calibrateServo(1);
  setSpeed(1, 0);
}

void robotLoop()
{
  robotPrintlnDebug();
  processBluetoothData();
  processData();
}

void processData()
{
  for(int dataOutput = 0; dataOutput < NUM_DATA_OUTPUTS; dataOutput++)
  {
    // If any of its inputs are set to autoPolling, get data from it and send to those inputs
    validGetData = false;
    for(int dataInput = 0; dataInput < DATA_OUTDEGREE && dataMapping[dataOutput][dataInput] >= 0; dataInput++)
    {
      if(autoPoll[dataOutput][dataInput])
      {
        if(!validGetData) // this output may have multiple attached inputs, but only request its data once
          getData(dataOutputIDs[dataOutput]);
        if(validGetData)
          processData(outputData, dataOutputIDs[dataOutput], dataMapping[dataOutput][dataInput]);
        else
          break;
      }
    }
  }
}

char* getData(int sourceID)
{
  return getData(sourceID, -1);
}

char* getData(int sourceID, int destID)
{
  if(sourceID == 7)
  {
    int input = (int) atof(getData(9));
    if(!validGetData)
    {
      outputData[0] = '\0';
      return outputData;
    }
    validGetData = true;
    itoa((int) (input), outputData, 10);
    return outputData;
  }
  if(getProtocol(sourceID) == PROTOCOL_BLUETOOTH)
  {
    // TODO control frequency of polling in a way that is fair to all polled inputs
    // and that also does not limit frequency of incoming requests
  
    // Request data from bluetooth
    // use format GET$outputPortID$inputPortID
    char toSend[20]; toSend[0] = '\0';
    strcpy(dataRequest, toSend, 20);
    concat(toSend, "$", toSend, 20);
    concatInt(toSend, sourceID, toSend, 20);
    concat(toSend, "$", toSend, 20);
    sendBluetoothData(toSend);
    // For now, pretend like it was an invalid getData
    // When the bluetooth controller responds with DATA command, it will be processed then
    outputData[0] = '\0';
    validGetData = false;
    return outputData;
  }
  if(sourceID == 17)
  {
    int input = (int) atof(getData(19));
    if(!validGetData)
    {
      outputData[0] = '\0';
      return outputData;
    }
    validGetData = true;
    itoa((int) (input), outputData, 10);
    return outputData;
  }
  outputData[0] = '\0';
  validGetData = false;
  return outputData;
}

void processData(const char* data, int sourceID, int* destIDs, int numDestIDs)
{
  for(int i = 0; i < numDestIDs; i++)
  {
    if(destIDs[i] >= 0)
      processData(data, sourceID, destIDs[i]);
  }
}

void processData(const char* data, int destID)
{
  return processData(data, -1, destID);
}

void processData(const char* data, int sourceID, int destID)
{
  if(destID == 0)
  {
    int speed = (int) atof(data);
    setSpeed(0, speed);
  }
  if(destID == 4)
  {
    int input = (int) atof(data);
    char outputData[10];
    itoa((int) (input), outputData, 10);
    for(int dataOutput = 0; dataOutput < NUM_DATA_OUTPUTS; dataOutput++)
    {
      if(dataOutputIDs[dataOutput] == 7)
        processData(outputData, dataOutput, dataMapping[dataOutput], DATA_OUTDEGREE);
    }
  }
  if(getProtocol(destID) == PROTOCOL_BLUETOOTH)
  {
    // TODO control frequency of polling in a way that is fair to all polled inputs
    // and that also does not limit frequency of incoming requests
  
    // Send data to bluetooth
    // use format DATA$data$outputPortID$inputPortID
    char toSend[50]; toSend[0] = '\0';
    strcpy(dataCommand, toSend, 50);
    concat(toSend, "$", toSend, 50);
    concat(toSend, data, toSend, 50);
    concat(toSend, "$", toSend, 50);
    concatInt(toSend, sourceID, toSend, 50);
    concat(toSend, "$", toSend, 50);
    concatInt(toSend, destID, toSend, 50);
    sendBluetoothData(toSend);
  }
  if(destID == 10)
  {
    int speed = (int) atof(data);
    setSpeed(1, speed);
  }
  if(destID == 14)
  {
    int input = (int) atof(data);
    char outputData[10];
    itoa((int) (input), outputData, 10);
    for(int dataOutput = 0; dataOutput < NUM_DATA_OUTPUTS; dataOutput++)
    {
      if(dataOutputIDs[dataOutput] == 17)
        processData(outputData, dataOutput, dataMapping[dataOutput], DATA_OUTDEGREE);
    }
  }
  robotPrintDebug("Finished ProcessData <");
  robotPrintDebug(data);
  robotPrintDebug(">");
  robotPrintDebug(" for ");
  robotPrintlnDebug(destID);
}